import 'dart:convert';
import 'dart:io';

import 'package:file_picker/file_picker.dart';
import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:viserpay/core/helper/date_converter.dart';
import 'package:viserpay/core/utils/my_strings.dart';
import 'package:viserpay/core/utils/status.dart';
import 'package:viserpay/core/utils/util.dart';
import 'package:viserpay/data/model/authorization/authorization_response_model.dart';
import 'package:viserpay/data/model/global/response_model/response_model.dart';
import 'package:viserpay/data/model/kyc/kyc_response_model.dart';
import 'package:viserpay/data/model/nominee/nominee_create_response.dart';
import 'package:viserpay/data/model/nominee/nominee_list_response_model.dart';
import 'package:viserpay/data/model/nominee/nominee_response_model.dart';
import 'package:viserpay/data/model/savings/savings_response_model.dart';
import 'package:viserpay/data/repo/nominee/nominee_repo.dart';
import 'package:viserpay/view/components/snack_bar/show_custom_snackbar.dart';

class NomineeController extends GetxController {
  final NomineeRepo nomineeRepo;
  NomineeController({required this.nomineeRepo});

  TextEditingController nameController = TextEditingController();
  TextEditingController birthDateController = TextEditingController();
  File? imageFile;

  bool isLoading = false;
  List<Nominee> nomineeList = [];
  String nomineeImagePath = "";
  Future<void> initialize() async {
    nomineeFormList = [];
    selectedNominee = null;
    isLoading = true;
    update();
    await getNomineeForm();
  }

  void updateNomineeListAndImagePath(List<Nominee> list, {String? imagePath}) {
    nomineeList = list;
    nomineeImagePath = imagePath ?? "";
    update();
    printX(nomineeList);
    printX("nomineeImagePath: $nomineeImagePath");
  }

  List<FormModel> nomineeFormList = [];
  Future<void> getNomineeForm() async {
    nomineeFormList = [];
    isLoading = true;
    update();
    try {
      ResponseModel response = await nomineeRepo.getNomineeForm();
      if (response.statusCode == 200) {
        NomineeResponseModel model = NomineeResponseModel.fromJson(jsonDecode(response.responseJson));
        if (model.status == "success") {
          nomineeFormList = model.data?.formData?.list ?? [];
        }
      }
    } catch (e) {
      printX(e);
    } finally {
      isLoading = false;
      update();
    }
  }

  Future<void> getNomineeList() async {
    nomineeList = [];
    nomineeFormList = [];
    isLoading = true;
    update();
    try {
      ResponseModel response = await nomineeRepo.getNomineeList();
      if (response.statusCode == 200) {
        NomineeListResponseModel model = NomineeListResponseModel.fromJson(jsonDecode(response.responseJson));
        if (model.status == "success") {
          selectedNominee = null;
          nomineeList = model.data ?? [];
        }
      }
    } catch (e) {
      printX(e);
    } finally {
      isLoading = false;
      update();
    }
  }

  bool isSubmitLoading = false;
  Future<void> addNewNominee() async {
    if (birthDateController.text.isEmpty) {
      CustomSnackBar.error(errorList: ["Nominee Birth Date is required"]);
      return;
    }
    if (nameController.text.isEmpty) {
      CustomSnackBar.error(errorList: ["Nominee Name is required"]);
      return;
    }
    if (imageFile == null) {
      CustomSnackBar.error(errorList: ["Nominee Image is required"]);
      return;
    }
    isSubmitLoading = true;
    update();
    try {
      NomineeCreateResponseModel response = await nomineeRepo.addNewNominee(
        list: nomineeFormList,
        birthDate: birthDateController.text,
        name: nameController.text,
        nomineeImage: imageFile!,
      );
      if (response.status == "success") {
        nomineeFormList.clear();
        imageFile = null;
        nameController.text = "";
        birthDateController.text = "";
        if (response.data != null) {
          nomineeList.add(response.data!);
        }
        Get.back();
        CustomSnackBar.success(successList: response.message?.success ?? [MyStrings.success]);
      } else {
        CustomSnackBar.error(errorList: response.message?.error ?? [MyStrings.somethingWentWrong]);
      }
    } catch (e) {
      printX(e);
    } finally {
      isSubmitLoading = false;
      update();
    }
  }

// select nominee
  Nominee? selectedNominee;
  void selectNominee(Nominee nominee) {
    selectedNominee = nominee;
    update();
  }

  // adding dynamic value
  void changeSelectedValue(value, int index) {
    nomineeFormList[index].selectedValue = value;
    update();
  }

  void changeSelectedRadioBtnValue(int listIndex, int selectedIndex) {
    nomineeFormList[listIndex].selectedValue = nomineeFormList[listIndex].options?[selectedIndex];
    update();
  }

  void changeSelectedCheckBoxValue(int listIndex, String value) {
    List<String> list = value.split('_');
    int index = int.parse(list[0]);
    bool status = list[1] == 'true' ? true : false;

    List<String>? selectedValue = nomineeFormList[listIndex].cbSelected;

    if (selectedValue != null) {
      String? value = nomineeFormList[listIndex].options?[index];
      if (status) {
        if (!selectedValue.contains(value)) {
          selectedValue.add(value!);
          nomineeFormList[listIndex].cbSelected = selectedValue;
          update();
        }
      } else {
        if (selectedValue.contains(value)) {
          selectedValue.removeWhere((element) => element == value);
          nomineeFormList[listIndex].cbSelected = selectedValue;
          update();
        }
      }
    } else {
      selectedValue = [];
      String? value = nomineeFormList[listIndex].options?[index];
      if (status) {
        if (!selectedValue.contains(value)) {
          selectedValue.add(value!);
          nomineeFormList[listIndex].cbSelected = selectedValue;
          update();
        }
      } else {
        if (selectedValue.contains(value)) {
          selectedValue.removeWhere((element) => element == value);
          nomineeFormList[listIndex].cbSelected = selectedValue;
          update();
        }
      }
    }
  }
  //

  void pickFile(int index, {bool isImageFile = false}) async {
    FilePickerResult? result = await FilePicker.platform.pickFiles(allowMultiple: false, type: FileType.custom, allowedExtensions: ['jpg', 'png', 'jpeg', 'pdf', 'doc', 'docx']);

    if (result == null) return;

    if (isImageFile) {
      imageFile = File(result.files.single.path!);
    } else {
      nomineeFormList[index].imageFile = File(result.files.single.path!);
      String fileName = result.files.single.name;
      nomineeFormList[index].selectedValue = fileName;
    }
    update();
    return;
  }

  void changeSelectedDateTimeValue(int index, BuildContext context) async {
    DateTime? pickedDate = await showDatePicker(
      context: context,
      initialDate: DateTime.now(),
      firstDate: DateTime(2000),
      lastDate: DateTime(2101),
    );
    if (pickedDate != null) {
      TimeOfDay? pickedTime = await showTimePicker(
        context: context,
        initialTime: TimeOfDay.now(),
      );
      if (pickedTime != null) {
        final DateTime selectedDateTime = DateTime(
          pickedDate.year,
          pickedDate.month,
          pickedDate.day,
          pickedTime.hour,
          pickedTime.minute,
        );

        nomineeFormList[index].selectedValue = DateConverter.estimatedDateTime(selectedDateTime);
        nomineeFormList[index].textEditingController?.text = DateConverter.estimatedDateTime(selectedDateTime);

        update();
      }
    }

    update();
  }

  void changeSelectedDateOnlyValue(int index, BuildContext context, {bool isBirthDate = false}) async {
    DateTime? pickedDate = await showDatePicker(
      context: context,
      initialDate: DateTime.now(),
      firstDate: DateTime(1900),
      lastDate: DateTime.now(),
    );
    if (pickedDate != null) {
      final DateTime selectedDateTime = DateTime(
        pickedDate.year,
        pickedDate.month,
        pickedDate.day,
      );

      if (isBirthDate) {
        birthDateController.text = DateConverter.estimatedDateYMD(selectedDateTime);
      } else {
        nomineeFormList[index].selectedValue = DateConverter.estimatedDate(selectedDateTime);
        nomineeFormList[index].textEditingController?.text = DateConverter.estimatedDate(selectedDateTime);
      }
    }

    update();
  }

  void changeSelectedTimeOnlyValue(int index, BuildContext context) async {
    TimeOfDay? pickedTime = await showTimePicker(
      context: context,
      initialTime: TimeOfDay.now(),
    );
    if (pickedTime != null) {
      final DateTime selectedDateTime = DateTime(
        DateTime.now().year,
        DateTime.now().month,
        DateTime.now().day,
        pickedTime.hour,
        pickedTime.minute,
      );

      nomineeFormList[index].selectedValue = DateConverter.estimatedTime(selectedDateTime);
      nomineeFormList[index].textEditingController?.text = DateConverter.estimatedTime(selectedDateTime);
      update();
    }

    update();
  }

  TextEditingController pinController = TextEditingController();
  FocusNode pinFocusNode = FocusNode();
  bool isCloseLoading = false;
  void removeNominee(String nomineeId) async {
    isCloseLoading = true;
    update();
    try {
      ResponseModel responseModel = await nomineeRepo.deleteNominee(nomineeId: nomineeId, pin: pinController.text);
      if (responseModel.statusCode == 200) {
        Get.back();
        AuthorizationResponseModel model = AuthorizationResponseModel.fromJson(jsonDecode(responseModel.responseJson));
        if (model.status == "success") {
          nomineeList.removeWhere((ele) => ele.id == nomineeId);
          if (selectedNominee?.id == nomineeId) {
            selectedNominee = null;
          }
          update();
          CustomSnackBar.success(successList: model.message?.success ?? [MyStrings.success]);
        } else {
          CustomSnackBar.error(errorList: model.message?.error ?? [MyStrings.somethingWentWrong]);
        }
      } else {
        CustomSnackBar.error(errorList: [responseModel.message]);
      }
    } catch (e) {
      printX(e);
    } finally {
      isCloseLoading = false;
      pinController.text = "";
      update();
    }
  }
}
